<?php
namespace App\Controllers;

use App\Models\DB;
use function App\Helpers\csrf_field;
use function App\Helpers\csrf_verify;
use function App\Helpers\audit;
use function App\Helpers\redirect;

class EventsController {

    public static function index(): string {
        $pdo = DB::pdo();
        $sql = "SELECT Id, Title, Location, DATE(StartDateTime) as StartDate, AllDay FROM events ORDER BY StartDateTime DESC, Title";
        $rows = $pdo->query($sql)->fetchAll();
        ob_start();
        $title = 'Events';
        include __DIR__ . '/../Views/events/list.php';
        return ob_get_clean();
    }

    public static function create(): string {
        $resource = [];
        ob_start();
        $title = 'New Events';
        include __DIR__ . '/../Views/events/form.php';
        return ob_get_clean();
    }

    
public static function store(): void {
    \App\Helpers\csrf_verify();

    $pdo = DB::pdo();
    $createdBy = \App\Helpers\current_username();
    $pdo->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(\PDO::ATTR_EMULATE_PREPARES, true);

//        'StartDateTime' => self::normDateTime($_POST['StartDateTime'] ?? null), // 'Y-m-d H:i:s' or null
//        'EndDateTime'   => self::normDateTime($_POST['EndDateTime'] ?? null),
    $data = [
        'Title'         => trim((string)($_POST['Title'] ?? '')),
        'Body'          => trim((string)($_POST['Body'] ?? '')),
        'Location'      => trim((string)($_POST['Location'] ?? '')),
        'StartDateTime' => $_POST['StartDate'] ?? null, // 'Y-m-d H:i:s' or null
        'EndDateTime'   => $_POST['EndDate'] ?? null,
        'AllDay'        => isset($_POST['AllDay']) ? 1 : 0,
        'BoardId'       => ($_POST['BoardId'] ?? '') !== '' ? (int)$_POST['BoardId'] : null,
        'CreatedAt'     => self::normDateTime($_POST['CreatedAt'] ?? null) ?: gmdate('Y-m-d H:i:s'),
        'CreatedBy' => $createdBy
    ];

    // Use plain placeholders; let NULLs be NULL.
    // Also: match the actual table name casing used on disk (Linux is case-sensitive).
    $sql = "INSERT INTO events
              (Title, Body, Location, StartDateTime, EndDateTime, AllDay, BoardId, CreatedAt, CreatedBy)
            VALUES
              (?, ?, ?, ?, ?, ?, ?, ?, ?)";

    $params = [
        $data['Title'],
        $data['Body'],
        $data['Location'],
        $data['StartDateTime'],   // null OK if column allows NULL
        $data['EndDateTime'],     // null OK if column allows NULL
        (int)$data['AllDay'],
        $data['BoardId'],         // null OK if column allows NULL
        $data['CreatedAt'],
        $data['CreatedBy'],
    ];

    try {
        \App\Helpers\log_info("Events.store SQL: {$sql}");
        \App\Helpers\log_info("Events.store params: " . json_encode($params, JSON_UNESCAPED_SLASHES));

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);

        // MySQL: use lastInsertId()
        $id = (int)$pdo->lastInsertId();

        \App\Helpers\audit('Events', 'create', $id, $data);
        \App\Helpers\redirect('/events');

    } catch (\Throwable $e) {
        \App\Helpers\log_error($e);
        if (isset($stmt)) {
            \App\Helpers\log_info('PDO errorInfo: ' . json_encode($stmt->errorInfo()));
        }
        http_response_code(500);
        echo "Save failed.";
    }
}

    private static function normDateTime(?string $v): ?string {
        $v = trim((string)($v ?? ''));
        if ($v === '') return null;
        $v = str_replace('T',' ', $v);
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $v)) {
            $v .= ' 00:00:00';
        } elseif (preg_match('/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}$/', $v)) {
            $v .= ':00';
        }
        if (!preg_match('/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}$/', $v)) {
            $ts = strtotime($v);
            $v = $ts ? gmdate('Y-m-d H:i:s', $ts) : null;
        }
        return $v ?: null;
    }

    public static function edit(): string {
        $id = (int)($_GET['id'] ?? 0);
        $pdo = DB::pdo();

        $stmt = $pdo->prepare("SELECT e.*, DATE(e.StartDateTime) as StartDate, DATE(e.EndDateTime) as EndDate FROM events e WHERE e.Id=?");
        $stmt->execute([$id]);
        $model = $stmt->fetch();

        if (!$model) {
            http_response_code(404);
            return "Not found";
        }

        ob_start();
        $title = 'Edit Events';
        include __DIR__ . '/../Views/events/form.php';
        return ob_get_clean();
    }

    public static function update(): void {
        \App\Helpers\csrf_verify();

        $id  = (int)($_POST['Id'] ?? 0);
        $pdo = DB::pdo();
        $pdo->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
        $pdo->setAttribute(\PDO::ATTR_EMULATE_PREPARES, true);

        $sel = $pdo->prepare("SELECT * FROM events WHERE Id=?");
        $sel->execute([$id]);
        $before = $sel->fetch() ?: [];

//            'StartDateTime' => self::normDateTime($_POST['StartDateTime'] ?? null),
//            'EndDateTime'   => self::normDateTime($_POST['EndDateTime'] ?? null),

        $data = [
            'Title'         => trim((string)($_POST['Title'] ?? '')),
            'Body'          => trim((string)($_POST['Body'] ?? '')),
            'Location'      => trim((string)($_POST['Location'] ?? '')),
            'StartDateTime' => $_POST['StartDate'] ?? null,
            'EndDateTime'   => $_POST['EndDate'] ?? null,
            'AllDay'        => isset($_POST['AllDay']) ? 1 : 0,
            'BoardId'       => isset($_POST['BoardId']) && $_POST['BoardId'] !== '' ? (int)$_POST['BoardId'] : null,
        ];

        $sql = "UPDATE events
                   SET Title=?,
                       Body=?,
                       Location=?,
                       StartDateTime=?,
                       EndDateTime=?,
                       AllDay=?,
                       BoardId=?
                 WHERE Id=?";

        $params = [
            $data['Title'],
            $data['Body'],
            $data['Location'],
            $data['StartDateTime'],
            $data['EndDateTime'],
            (int)$data['AllDay'],
            $data['BoardId'],
            $id,
        ];

        try {
            \App\Helpers\log_info("Events.update SQL: {$sql}");
            \App\Helpers\log_info("Events.update params: " . json_encode($params, JSON_UNESCAPED_SLASHES));

            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);

            $sel->execute([$id]);
            $after = $sel->fetch() ?: [];

            \App\Helpers\audit('Events', 'update', $id, ['before' => $before, 'after' => $after]);
            \App\Helpers\redirect('/events');

        } catch (\Throwable $e) {
            \App\Helpers\log_error($e);
            if (isset($stmt)) {
                \App\Helpers\log_info('PDO errorInfo: ' . json_encode($stmt->errorInfo()));
            }
            http_response_code(500);
            echo "Update failed.";
        }
    }

    public static function delete(): void {
        csrf_verify();
        $id = (int)($_POST['id'] ?? 0);
        $pdo = DB::pdo();
        $sel = $pdo->prepare("SELECT * FROM events WHERE Id=?");
        $sel->execute([$id]);
        $row = $sel->fetch() ?: [];
        $stmt = $pdo->prepare("DELETE FROM events WHERE Id=?");
        $stmt->execute([$id]);
        audit('Events','delete',$id,['before'=>$row]);
        redirect('/events');
    }
}

