<?php
namespace App\Helpers;

    /**
     * Save an uploaded file to uploads/YYYY/{table}/ and return its relative path + final name.
     *
     * @param string $inputName   The name of the <input type="file"> (e.g. "attachment")
     * @param string $parentTable Used to name the destination folder
     * @return array|null         ['FilePath' => 'uploads/...', 'FileName' => 'file.ext'] or null if nothing uploaded
     * @throws \RuntimeException  If move_uploaded_file fails
     */
    function handleupload(string $inputName, string $parentTable): ?array
    {
        if (empty($_FILES[$inputName]['name']) || !is_uploaded_file($_FILES[$inputName]['tmp_name'])) {
            return null; // nothing uploaded
        }
        \App\Helpers\log_info("parentTable: {$parentTable}");

        $origName = $_FILES[$inputName]['name'];
        $tmpPath  = $_FILES[$inputName]['tmp_name'];
        \App\Helpers\log_info("origName: {$origName}");
        \App\Helpers\log_info("tmpPath: {$tmpPath}");

        // Normalize folder path: uploads/YYYY/tablename/
        $year = date('Y');
        $safeTable = preg_replace('/[^A-Za-z0-9_\-]/', '', $parentTable);
        \App\Helpers\log_info("safeTable: {$safeTable}");

        $baseDir = "../../uploads/{$year}/{$safeTable}";
        #$baseDir = dirname(__DIR__, 2) . "/uploads/{$year}/{$safeTable}";
        \App\Helpers\log_info("baseDir: {$baseDir}");
        if (!is_dir($baseDir)) {
            mkdir($baseDir, 0775, true);
        }

        // Sanitize filename
        $base = pathinfo($origName, PATHINFO_FILENAME);
        $ext  = strtolower(pathinfo($origName, PATHINFO_EXTENSION));
        $base = iconv('UTF-8', 'ASCII//TRANSLIT', $base);
        $base = preg_replace('/[^A-Za-z0-9_\-]+/', '-', $base);
        $base = trim($base, '-_') ?: 'file';

        // Collision-free name
        $n = 0;
        do {
            $suffix = $n === 0 ? '' : "-{$n}";
            $finalName = "{$base}{$suffix}" . ($ext ? ".{$ext}" : '');
            $target = "{$baseDir}/{$finalName}";
            $n++;
        } while (file_exists($target));

        if (!move_uploaded_file($tmpPath, $target)) {
            throw new \RuntimeException("Failed to move uploaded file.");
        }

        $relPath = "uploads/{$year}/{$safeTable}/{$finalName}";
        return ['FilePath' => $relPath, 'FileName' => $finalName];
    }
